function [k_pooled, conc_corr, k_runs, k_pooled_draws] = ...
    Crocodile_model(Proteins, rate_par_draws, Data_table, varargin)
% The purpose of this function is fit the GTPase assay data with a GTPase
% supplied in the standardized data format using the CROCODILE model
% (CROwding, COoperativity and DImerization in Luminsecence Experiments).
% This function then returns the fit results, optionally incorporating an
% assay specific activity correction for the GTPase and optionally plotting
% and storing the model fits. If applicable, the interaction rate constant
% is tested, using a unequal variance t-test [1].
%
% Inputs:
% Protein     = cell array of strings with the varied proteins in the assay
% rate_par_draws = Draws of GTPase rate parameters (k1, k2)
% Data_table  = the output is a table with the following variables/columns:
%               Run (name) - Time ([h]) - GTP_remaining (normalized such
%               that 100% = 1) - Error (of GTP_remaining) -  Buffer_error -
%               GTPase_conc ([uM]) and then if applicable, X1_conc, X2_conc
%               , with X1, X2 as other protein(s) of interest, e.g., Cdc24.
% varargin    = optional inputs, variable number of arguments.
%               Input 1: logical that states whether to take into account
%               an assay-specific activity correction for GTPase activity,
%               if applicable (if the GTPase is not varied in the assay).
%               If not provided, the default is true.
%               Input 2: logical that states whether the fits should be
%               plotted and stored. If not provided, the default is true.
%               Input 3: logical that states whether some fit results
%               should be printed in the command window. If not provided,
%               the default is true.
%               Input 4: logical that states whether to exclude data points
%               that have less than 5% GTP remaining. I false, all data
%               points are considered. If not provided, the default is
%               true.
%               Input 5: two-element vector that state the lower and upper
%               bound respectively for the GTPase concentration factor
%               of points to be included in pooling of estimates (default
%               is [0 Inf]).
%               Input 6: logical that states whether the discard runs from
%               pooling that have very low standard errors (1e-10 or lower)
%               on the k values. If not provided, the default is true.

% Outputs:
% k_pooled    = Table of pooled rate parameters fit results with variable
%               names: 'est' (point estimates), 'std_err' (standard errors
%               estimates), 'conf_int' (95% confidence intervals of
%               estimates ordered as lower and upper bound parameter 1,
%               lower and upper bound parameter 2, etc.), 'num_obs' (number
%               of observations underlying the esimtates), 'deg_free'
%               (degrees of freedom underlying the estimates).
%               If there is one run, the pooled estimate is the individual
%               estimate. If there is more than one run, the estimates
%               incorporated in the pooled estimate have standard errors
%               above 1e-10, GTPase activity correction factors > 0.5 and
%               buffer errors < 10%.
% conc_corr   = nx4 matrix with n as the number of runs and in each column
%               the point estimate, 95% confidence interval (lower bound,
%               then upper bound) and standard error of the GTPase activity
%               correction factor).
% k_runs      = Table of rate parameters fit results per run with variable
%               names: 'name' (run name), 'est' (point estimates),
%               'conf_int' (95% confidence intervals of estimates ordered
%               as lower and upper bound parameter 1, lower and upper bound
%               parameter 2, etc.), 'std_err' (standard errors estimates),
%               'num_obs' (number of observations underlying the estimates)
%               , 'deg_free' (degrees of freedom underlying the estimates),
%               'Adj_R_sq' (Adjusted R-squared), 'Interaction_p' (p-value
%               t-test of non-zero interaction k3 if applicable),
%               'c_GTPase' (GTPase activity correction factor), and 'pool'
%               (logical indicating whether the estimate of this run
%               contributed to the pooled estimate).
% k_pooled_draws = Random draws of the pooled rate parameters (with draws
%               from k1, k2, k3,X1, k3,X2, and k3,X1-X2 in successive
%               columns.
%
% Date: 18-01-2024
%
% Bibliography
% [1]   Ruxton, G. D. The unequal variance t-test is an underused
%       alternative to Students t-test and the MannWhitney U test.
%       Behavioral Ecology 17, 688690 (2006).

% Reset random number generator for reproducible random draws of rate parameters
rng('default')
% Disable certain warnings
warning('off', 'stats:nlinfit:ModelConstantWRTParam')
warning('off', 'MATLAB:rankDeficientMatrix')
warning('off', 'stats:nlinfit:Overparameterized')

%% Depending on assay proteins, define fit functions, boundary conditions rate parameters and how to print outputs

% For the fit functions, the following principle applies:
% Every function is of the form [GTP remaining] = exp(-rate * time), with
% rate = k1 * [GTPase] + k2 * [GTPase] ^ 2 + k3,X1 * [GTPase] * [X1] ^ n1 + 
% k3,X2 * [GTPase] * [X2] ^ n2 + k3,X1-X2 * [GTPase] * [X1] ^ n1 * [X2] ^n2
% where X1 and X2 are proteins other than the GTPase varied in this assay
% if applicable (otherwise these terms are ignored).
% Exponents n1 and n2 are 1 (linear dependencies) except for Cdc24
% where the default is 2 (quadratic dependence).
% [GTPase] is the effective GTPase concentration, which may include an
% activity correction factor ([GTPase] = c_corr * [GTPase_reported]) for
% assays with constant GTPase, unless this is disabled.
% All factors/terms that are constant within the assay are merged where
% possible, so e.g., a Bem1 sweep has its rate reduced to a + b * [Bem1],
% with a = k1 * [GTPase] + k2 * [GTPase] ^ 2 and
% b = k3,Bem1 * c_corr * [GTPase_reported].
% The reduced parameters are then transformed to fitting parameters, which
% allows us to enforce certain restrictions. In practice the restrictions
% entail certain reduced parameters are non-negative.
%
% So: we estimate the fitting parameters, which we transform to the reduced
% parameters which encompass the model rate parameters.

GTPase_only             = isempty(Proteins.LinEff) && isempty(Proteins.QuaEff);
Proteins_var            = cat(1, Proteins.QuaEff, Proteins.LinEff);
if isempty(Proteins_var)
    Proteins_var        = Proteins.GTPase;
end
if GTPase_only  % In this case,
    % we are looking at a pure GTPase sweep (no other varied protein)
    % par_trans defines the transformation to fitting parameter, to enforce
    % restrictions on rate parameter values.
    % For example, the function exp(x) as the first element means we set
    % k1 = exp(c1), with c1 as a constant spanning the real line.
    % We fit c1, and then transform 'c1' back to k1, always resulting in a
    % positive number. By the same token, setting the funciton x as the
    % second element, we have the identity transformation so k2 = b and no
    % restrictions apply.
    par_trans           = {@(x) exp(x); @(x) x};
    % Fit function for the GTP remaining (exp(-rate * t), including the
    % transformation of rate parameter to fitting parameters.
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) .* ct(:, 1) + ...
                    par_trans{2}(par(2)) .* (ct(:, 1) .^ 2)) .* ct(:, 2));
    % Note: par are the fitting parameters, ct are the concentration values
    % in columns followed by the time in the last column. So here ct(:, 1)
    % is the GTPase concentration data column, ct(:, 2) is the time.
    % String that is completed with fit values later to print in the
    % command window (unless printing is disabled)
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-k1 * ([%s] + k2 * [%s] ^ 2))\n', ...
                            'k1 = %%0.3g (%%0.3g-%%0.3g), k2 = %%0.3g (%%0.3g-%%0.3g)\nAdj. R^2 = %%0.3f\n\n'), ...
                            Proteins.GTPase{1}, Proteins.GTPase{1});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = 'Parameter 1: k1,  Parameter 2: k2';
elseif isempty(Proteins.LinEff) && ~isempty(Proteins.QuaEff) && (numel(Proteins.QuaEff) == 1)
    % In this case, we are looking at a sweep with one quadratic effector
    % a = k1 * [GTPase] + k2 * [GTPase] ^ 2 must be positive
    % b = k3,qua_eff * [GTPase] (including activity correction factor) is unrestricted
    par_trans           = {@(x) exp(x); @(x) x};
    % Two fit functions are defined, the default has quadratic dependence
    % of the rate on [qua_eff]
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) + par_trans{2}(par(2)) .* ...
                            ct(:, 1) .^ 2) .* ct(:, 2));
    % String that is completed with fit values later to print in the
    % command window (unless printing is disabled)
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-(a + b * [%s] ^ 2) * t)\n', ...
                             'a = %%0.3g (%%0.3g-%%0.3g), b = %%0.3g (%%0.3g-%%0.3g),\nAdj. R^2 = %%0.3f\n'), ...
                             Proteins.QuaEff{1});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = sprintf('Parameter 1: k1 + k2 * [%s_reported],  Parameter 2: k3,%s', ...
                            Proteins.GTPase{1}, Proteins.QuaEff{1});
    
elseif ~isempty(Proteins.LinEff) && ~isempty(Proteins.QuaEff) && ...
        (numel(Proteins.LinEff) == 1) && (numel(Proteins.QuaEff) == 1)
    % In this case, a linear and quadratic effector protein are varied
    % a  = k1 * [GTPase] + k2 * [GTPase] ^ 2 must be positive
    % b1 = k3,qua_eff * [GTPase] is unrestricted
    % b2 = k3,lin_eff * [GTPase] is unrestricted
    % c = k3,qua_eff-lin_eff * [GTPase] is unrestricted
    par_trans           = {@(x) exp(x); @(x) x; @(x) x; @(x) x};
    % To ensure we know which column in the data matrix supplied to the fit
    % function belongs to which protein, we adhere to the order in variable
    % Protein
    ind_qua             = find(strcmp(Proteins_var, Proteins.QuaEff{1}));
    ind_lin             = setdiff(1 : 2, find(strcmp(Proteins_var, Proteins.QuaEff{1})));
    % Two fit functions are defined, the default has quadratic dependence
    % of the rate on [qua_eff]
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) + par_trans{2}(par(2)) .* ct(:, ind_qua) .^ 2 + ...
                            par_trans{3}(par(3)) .* ct(:, ind_lin) + ...
                            par_trans{4}(par(4)) .* ct(:, ind_qua) .^ 2 .* ct(:, ind_lin)) .* ct(:, 3));
    % String that is completed with fit values later to print in the
    % command window (unless printing is disabled)
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-(a + b * [%s] ^ 2\n', ...
                            '+ b3 * [%s] + c * [%s] ^ 2 * [%s]) * t)\n', ...
                            'a = %%0.3g (%%0.3g-%%0.3g), b1 = %%0.3g (%%0.3g-%%0.3g),\n', ...
                            'b2 = %%0.3g (%%0.3g-%%0.3g), c = %%0.3g (%%0.3g-%%0.3g)\n', ...
                            'Adj. R^2 = %%0.3f\n'), Proteins.QuaEff{1}, ...
                            Proteins.LinEff{1}, Proteins.QuaEff{1}, Proteins.LinEff{1});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = sprintf(strcat('Parameter 1: k1 + k2 * [%s_reported], Parameter 2: k3,%s', ...
                            ', Parameter 3: k3,%s, Parameter 4: k3,%s-%s'), Proteins.GTPase{1}, Proteins.QuaEff{1}, ...
                            Proteins.LinEff{1}, Proteins.QuaEff{1}, Proteins.LinEff{1});
elseif ~isempty(Proteins.LinEff) && isempty(Proteins.QuaEff) && ...
        (numel(Proteins.LinEff) == 2)
    % In this case, 2 linear effector proteins are varied
    % a  = k1 * [GTPase] + k2 * [GTPase] ^ 2 must be positive
    % b1 = k3,X1 * [GTPase] is unrestricted
    % b2 = k3,X2 * [GTPase] is unrestricted
    % c = k3,X1-X2 * [GTPase] is unrestricted
    par_trans           = {@(x) exp(x); @(x) x; @(x) x; @(x) x};
    % To ensure we know which column in the data matrix supplied to the fit
    % function belongs to which protein, we adhere to the order in variable
    % Protein
    % Fit function with linear dependence on X1 & X2
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) + par_trans{2}(par(2)) .* ct(:, 1) + ...
                            par_trans{3}(par(3)) .* ct(:, 2) + ...
                            par_trans{4}(par(4)) .* ct(:, 1) .* ct(:, 2)) .* ct(:, 3));
    % String that is completed with fit values later to print in the
    % command window (unless printing is disabled)
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-(a + b * [%s]\n', ...
                            '+ b3 * [%s] + c * [%s] * [%s]) * t)\n', ...
                            'a = %%0.3g (%%0.3g-%%0.3g), b1 = %%0.3g (%%0.3g-%%0.3g),\n', ...
                            'b2 = %%0.3g (%%0.3g-%%0.3g), c = %%0.3g (%%0.3g-%%0.3g)\n', ...
                            'Adj. R^2 = %%0.3f\n'), Proteins_var{1}, Proteins_var{2}, ...
                            Proteins_var{1}, Proteins_var{2});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = sprintf(strcat('Parameter 1: k1 + k2 * [%s_reported], Parameter 2: k3,%s', ...
                            ', Parameter 3: k3,%s, Parameter 4: k3,%s-%s'), Proteins.GTPase{1}, ...
                            Proteins_var{1}, Proteins_var{2}, ...
                            Proteins_var{1}, Proteins_var{2});
elseif isempty(Proteins.LinEff) && ~isempty(Proteins.QuaEff) && ...
        (numel(Proteins.QuaEff) == 2)
    % In this case, 2 quadratic effector proteins are varied
    % a  = k1 * [GTPase] + k2 * [GTPase] ^ 2 must be positive
    % b1 = k3,X1 * [GTPase] is unrestricted
    % b2 = k3,X2 * [GTPase] is unrestricted
    % c = k3,X1-X2 * [GTPase] is unrestricted
    par_trans           = {@(x) exp(x); @(x) x; @(x) x; @(x) x};
    % To ensure we know which column in the data matrix supplied to the fit
    % function belongs to which protein, we adhere to the order in variable
    % Protein
    % Fit function with linear dependence on X1 & X2
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) + par_trans{2}(par(2)) .* ct(:, 1) .^ 2 + ...
                            par_trans{3}(par(3)) .* ct(:, 2) .^ 2 + ...
                            par_trans{4}(par(4)) .* ct(:, 1) .^ 2 .* ct(:, 2) .^ 2) .* ct(:, 3));
    % String that is completed with fit values later to print in the
    % command window (unless printing is disabled)
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-(a + b * [%s] .^ 2 \n', ...
                            '+ b3 * [%s] .^ 2 + c * [%s] .^ 2 * [%s] .^ 2) * t)\n', ...
                            'a = %%0.3g (%%0.3g-%%0.3g), b1 = %%0.3g (%%0.3g-%%0.3g),\n', ...
                            'b2 = %%0.3g (%%0.3g-%%0.3g), c = %%0.3g (%%0.3g-%%0.3g)\n', ...
                            'Adj. R^2 = %%0.3f\n'), Proteins_var{1}, Proteins_var{2}, ...
                            Proteins_var{1}, Proteins_var{2});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = sprintf(strcat('Parameter 1: k1 + k2 * [%s_reported], Parameter 2: k3,%s', ...
                            ', Parameter 3: k3,%s, Parameter 4: k3,%s-%s'), Proteins.GTPase{1}, ...
                            Proteins_var{1}, Proteins_var{2}, ...
                            Proteins_var{1}, Proteins_var{2});
elseif ~isempty(Proteins.LinEff) && isempty(Proteins.QuaEff) && ...
        (numel(Proteins.LinEff) == 1)
    % In this case, we are looking at a linear effector sweep with GTPase
    % concentration fixed
    % a = k1 * [GTPase] + k2 * [GTPase] ^ 2 must be positive
    % b = k3,X1 * [GTPase] is unrestricted
    par_trans           = {@(x) exp(x); @(x) x};
    fit_fun             = @(par, ct) exp(- (par_trans{1}(par(1)) + par_trans{2}(par(2)) .* ct(:, 1)) .* ct(:, 2));
    print_fit           = sprintf(strcat('[GTP] = 100 * exp(-(a + b * [%s]) * t)\n', ...
                            'a = %%0.3g (%%0.3g-%%0.3g), b = %%0.3g (%%0.3g-%%0.3g),\nAdj. R^2 = %%0.3f\n'), Proteins.LinEff{1});
    % String to store to annotate how to interpret each parameter estimate
    par_legend          = sprintf(strcat('Parameter 1: k1 + k2 * [GTPase_reported], Parameter 2: k3,%s'), ...
                            Proteins.GTPase{1}, Proteins.LinEff{1});
end

% Define starting points for fit of fitting parameters
par_start                = cellfun(@(x, y) x(y), par_trans, num2cell(0.1 * ones(size(par_trans))));
if ~isempty(Proteins.QuaEff)
    % adjust starting point that worked poorly
    par_start(2)         = par_trans{2}(0.05);
    if numel(Proteins.QuaEff) == 2
        par_start(3)     = par_trans{3}(0.05);
    end
end
% Number of fitting parameters
num_par                 = numel(par_trans);

%% Extract parameters/settings from optional varargin argument:
% Logical that states whether to take into account an assay-specific
% activity correction for GTPase activity.
% Logical that states whether the fits should be plotted and stored.
% Logical that states whether some fit results should be printed.
% Logical that states whether to exclude data points with <=5% GTP remaining
% Two-element vector that indicates bounds for the GTPase concentration
% correction factor of runs to be included in pooling
% Logical that states whether to exclude data points from pooling with very
% low (<=1e-10) standarderors on k values)
[act_corr, plot_fits, print_fits, GTP_filt] = deal(true(1));
conc_corr_bounds    = [0 Inf];
k_low_err_filt      = true;
if numel(varargin) >= 1
    act_corr            = varargin{1};
end
if numel(varargin) >= 2
    plot_fits           = varargin{2};
end
if numel(varargin) >= 3
    print_fits          = varargin{3};
end
if numel(varargin) == 4
    GTP_filt            = varargin{4};
end
if numel(varargin) == 5
    conc_corr_bounds    = varargin{5};
end
if numel(varargin) == 6
    k_low_err_filt      = varargin{6};
end

% Show in the command window the assay type (which proteins are varied)
if print_fits
    fprintf('\n%s\n%s%s\n', repmat('-', 1, 20), sprintf('%s\n', Proteins_var{:}), repmat('-', 1, 20))
end

%% Last preparations for loop over runs
% Find the unique run names
run_names               = table2cell(unique(Data_table(:, 1), 'stable'));
% Preallocate variables for results
fig_rate_fit            = gobjects(numel(run_names), numel(Proteins_var));
[Adj_R_sq, buf_err, Interaction_p, c_GTPase] = deal(NaN(numel(run_names), 1));
conc_corr               = NaN(numel(run_names), 4);
k_val                   = NaN(numel(run_names), 4 * num_par + 1);

for r = 1 : numel(run_names) % Loop over all runs

%% Extract data per run, perform filtering if required
    
    % Find the data rows corresponding to this particular run
    inds_run            = find(strcmp(Data_table.Run, run_names{r}));
    % Find the GTPase concentrations
    GTPase_conc         = table2array(Data_table(inds_run, [Proteins.GTPase{1} '_conc']));
    % Find all varied protein concentrations
    conc                = table2array(Data_table(inds_run, strcat(Proteins_var(:), '_conc')));
    % For plotting the fit, define the concentration axis along which the
    % fit is evaluated in x_probe
    conc_fit            = linspace(0, max(Data_table.(strcat(Proteins_var{1}, '_conc'))) + 0.1, 100)';
    if numel(Proteins_var) > 1
        % In case there is more than one varied protein, also multidimensional concentration axis to evaluate fit
        conc2_uni       = unique(conc(:, 2));
        conc_fit        = [repmat(conc_fit, numel(conc2_uni), 1) ...
                            repelem(conc2_uni, repmat(numel(conc_fit), numel(conc2_uni), 1), 1)];
        % Also plot multidimensional concentration axis (with other protein on x-axis)
        conc_fit2       = linspace(0, max(Data_table.(strcat(Proteins_var{2}, '_conc'))) + 0.1, 100)';
        conc2_uni2      = unique(conc(:, 1));
        conc_fit2       = [repelem(conc2_uni2, repmat(numel(conc_fit2), numel(conc2_uni2), 1), 1) ...
                            repmat(conc_fit2, numel(conc2_uni2), 1) ];
    end
    % GTP remaining
    GTP                 = Data_table.GTP_remaining(inds_run);
    % Data weights for the fits (less if the error is large)
    weight              = 1 ./ Data_table.Error(inds_run);
    % Time points
    time                = Data_table.Time(inds_run);
    % Buffer error, if available
    if isfield(table2struct(Data_table), 'Buffer_error')
        buf_err(r)      = Data_table.Buffer_error(inds_run(1));
    end
    
    % Filter data with low GTP remaining if required
    if GTP_filt
        ind_filt        = GTP > 0.05;
    else
        ind_filt        = true(size(GTP));
    end

    % If the parameters will not be able to be fitted due to lack of data,
    % stop evaluating this run
    if (nnz(ind_filt) < num_par) || (numel(unique(conc)) < num_par) || all(isnan(conc(:)))
        continue
    end
    
%% Fitting rates, fitting parameters and their simulated draws
    % Update the starting points for the optimization of fit parameters
    % with fminsearch first
    par_start2          = fminsearch(@(pars) dist(GTP(ind_filt)', fit_fun(pars, [conc(ind_filt, :) time(ind_filt)])), ...
                            par_start, optimset('MaxFunEvals', 1e4));
    % Actual fit routine
    fit_out             = fitnlm([conc time], GTP, fit_fun, par_start2, 'Weights', weight .^ 2, 'Exclude', ~ind_filt);
    % Extract fit parameters including covariance
    pars                = fit_out.Coefficients.Estimate;
    if nnz(ind_filt) > num_par
        cov_pars        = fit_out.CoefficientCovariance;
    else
        cov_pars        = zeros(num_par);
    end
    % Generate random draws for the fitting parameters
    % Note, this normal approximation works best with a large number of observations.
    % One can think of improving this for more appropriate multivariate draws
    par_draws           = mvnrnd(pars, cov_pars, numel(rate_par_draws.k1));
    % Transform the random draws back to the reduced model parameters
    par_draws_t         = cell2mat(cellfun(@(x, y) x(y), par_trans', mat2cell(par_draws, size(par_draws, 1), ...
                            ones(1, num_par)), 'UniformOutput', false));
   
    % Caclulate rates based on estimated model parameters
    rate_fit            = -log(fit_out.predict([conc_fit ones(size(conc_fit, 1), 1)]));
    if numel(Proteins_var) > 1
        % In case there is more than one varied protein, also rate fit with other protein on x-axis
        rate_fit2       = -log(fit_out.predict([conc_fit2 ones(size(conc_fit2, 1), 1)]));
    end
    % y_fit is time-independent rate, so we just supply a default time of 1.
    Adj_R_sq(r)         = fit_out.Rsquared.Adjusted;
    
    % Print in the command window the fit results
    if print_fits
        fprintf(strcat(run_names{r}, ': ', print_fit, '\n'),  quantile(par_draws_t, [0.5 0.025 0.975]), Adj_R_sq(r));
    end
    
%% Conversion of fitting parameters back to model parameters

    % [GTPase_reported] for conversion of reduced parameters to rate parameters
    % ([GTPase] = correction factor * [GTPase_reported])
    if GTPase_only
        c_GTPase(r)     = 1;
        % In the case the GTPase is varied, we already take into account
        % [GTPase] explicitly in the reduced model parameters, so we set
        % the transformation factor to unity.
    else
        c_GTPase(r)     = mode(GTPase_conc);
        % In this case, the reduced parameters absorbed the constant [GTPase],
        % so we need to divide out the [GTPase_reported] from the reduced
        % parameters to retrieve the rate parameters.
    end
    
    % Calculate GTPase activity correction factor, if applicable
    % (not in a GTPase sweep and not if the user has set this correction to false)
    if GTPase_only || ~act_corr
        conc_corr(r, :) = [1 1 1 0];
    else % Generate the random draws for the correction factor based on the random draws of k1 and k2 calculated earlier
        conc_corr_draws = (-rate_par_draws.k1 + sqrt(rate_par_draws.k1 .^ 2 + 4 * par_draws_t(:, 1) .* rate_par_draws.k2)) ./ ...
                            ( 2 * rate_par_draws.k2 * c_GTPase(r) );
        % Note that par_draws_t(:, 1) is the first reduced parameter,
        % which estimates k1 * [GTPase] + k2 * [GTPase] ^ 2 for this run.
        conc_corr_draws(imag(conc_corr_draws) ~= 0 | conc_corr_draws < 0.1) = 0.1;
        conc_corr_draws(conc_corr_draws > 10) = 10;
        % Assume the correction factor cannot be below 0.1 or above 10
        % Find the correction factor with the 95% confidence interval and
        % standard error
        conc_corr(r, :) = [quantile(conc_corr_draws, [0.5 0.025 0.975]) std(conc_corr_draws)];
        % Show results for the correction factor in the command window
        if print_fits
            fprintf('GTPase concentration correction factor: %0.2f (%0.2f-%0.2f)\n', conc_corr(r, 1 : 3)');
        end
    end
    
    % We now extract the rate parameters from the reduced parameters by
    % dividing out the [GTPase] from the reduced parameter random draws
    k_draws             = par_draws_t / (conc_corr(r, 1) * c_GTPase(r));
    % Note: the first parameter in the case of a assay where the GTPase is not
    % varied is k1 + k2 * [GTPase]
    k_val(r, :)         = [reshape([quantile(k_draws, [0.5 0.025 0.975]); std(k_draws)], [], 1)' nnz(ind_filt)];

    % p-value for t-test for testing non-zero interaction k3 (k3,X1-X2)
    if num_par == 4
        Interaction_p(r)    = 1 - tcdf(abs(k_val(r, 13) / k_val(r, 16)), k_val(r, 17) - 1);
    end
    
%% Preparation for plots (and labels)
    if plot_fits
        % Define legend text (time points when the GTPase is varied,
        % GTPase concentration when 1 protein is varied, concentrations of
        % the second protein if 2 proteins are varied
        if GTPase_only || (numel(unique(time)) > 1)
            le_text     = arrayfun(@(x) sprintf('t = %0.f min.', x), 60 * unique(time), 'UniformOutput', false);
        elseif numel(Proteins_var) > 1
            le_text     = cellfun(@(x) sprintf('%s \\muM %s', x, Proteins_var{2}), cellstr(num2str(conc2_uni)), 'UniformOutput', false)';
            le_text2    = cellfun(@(x) sprintf('%s \\muM %s', x, Proteins_var{1}), cellstr(num2str(conc2_uni2)), 'UniformOutput', false)';
        else
            le_text     = {sprintf('%s \\muM %s', num2str(c_GTPase(r)), Proteins.GTPase{1})};
        end
        % Make title for the plots
        title_plot          = sprintf('Run %s', Data_table.Run{inds_run(1)}(2 : end));
        % Generate figure with the fit
        fig_rate_fit(r, 1)  = plot_rate_data(conc_fit, rate_fit, conc, GTP, 1./weight, time, Proteins_var, ...
                                le_text, title_plot, ind_filt, Adj_R_sq(r));
        if numel(Proteins_var) > 1
            fig_rate_fit(r, 2)  = plot_rate_data(fliplr(conc_fit2), rate_fit2, fliplr(conc), GTP, 1./weight, ...
                                    time, flipud(Proteins_var), le_text2, title_plot, ind_filt, Adj_R_sq(r));
        end
    end
    if print_fits
        fprintf('\n');
    end
end

%% Define outputs (pool by regression if needed and draw from Student t-distributions)
% Gather fit results divided by run
k_runs               = table(run_names, k_val(:, 1 : 4 : end - 1), k_val(:, unique([2 : 4 : end - 2 3 : 4 : end - 2])), ...
                        k_val(:, 4 : 4 : end - 1), k_val(:, end), k_val(:, end) - num_par, Adj_R_sq, Interaction_p, c_GTPase, ...
                        repmat({par_legend}, numel(run_names), 1), ...
                        'VariableNames', {'name'; 'est'; 'conf_int'; 'std_err'; 'num_obs'; 'deg_free';
                        'Adj_R_sq'; 'Interaction_p'; 'c_GTPase'; 'par_interpretation'});

% Define which runs to incorporate in the pooled estimate. If there is one
% run, the pooled estimate is the individual estimate. If there is more
% than one run, the estimates incorporated in the pooled estimate have
% standard errors above 1e-10 (otherwise the model parameter is possibly
% hitting the restriction bounds), GTPase activity correction factors > 0.5
% (otherwise there is an indication that there was something wrong with the
% run) and buffer errors < 10% (otherwise indicative that multiple aliquots
% of buffer where used to determine the buffer error which propagate to the
% standard errors of the other data points as well).
if numel(run_names) > 1
    ind_filt            = (conc_corr(:, 1) > conc_corr_bounds(1)) & ...
                            (conc_corr(:, 1) < conc_corr_bounds(2)) & (buf_err < 0.1);
    if k_low_err_filt
        ind_filt        = ind_filt & all(k_runs.std_err > 1e-10, 2);
    end
else
    ind_filt            = true(1);
end
% The pooled estimate follows from a wighted regression without intercept
% (line through all points)
lin_fit                 = cell(1, num_par);
for p = 1 : num_par
    if ~any(ind_filt)
        lin_fit{p}      = fitlm(ones(numel(ind_filt), 1), k_runs.est(:, p), 'Intercept', false);
    else
        lin_fit{p}      = fitlm(ones(numel(ind_filt), 1), k_runs.est(:, p), ...
                           'Weights', 1 ./ k_runs.std_err(:, p) .^ 2, 'Intercept', false, 'Exclude', ~ind_filt);
    end
end
k_runs.pool             = ind_filt;

% Store results in the pooled variable
if nnz(ind_filt) == 1
    k_pooled            = k_runs(ind_filt, :);
else
    k_pooled.est        = cellfun(@(x) x.Coefficients.Estimate, lin_fit);
    k_pooled.std_err    = cellfun(@(x) x.Coefficients.SE, lin_fit);
    k_pooled.conf_int   = cell2mat(cellfun(@(x) coefCI(x, 0.05), lin_fit, 'UniformOutput', false));
    k_pooled.num_obs    = lin_fit{1}.NumObservations;
    k_pooled.deg_free   = lin_fit{1}.DFE;
    k_pooled.par_interpretation = par_legend;
end

% Generate random draws of the pooled estimate
k_pooled_draws          = bsxfun(@plus, k_pooled.est, bsxfun(@times, k_pooled.std_err, ...
                            trnd(k_pooled.deg_free, numel(rate_par_draws.k1), numel(k_pooled.est))));

% p-value for t-test for testing non-zero interaction pooled k3 (k3,X1-X2)
if num_par == 4  
    k_pooled.Interaction_p  = 1 - tcdf(abs(k_pooled.est(4) / k_pooled.std_err(4)), k_pooled.num_obs);
else
    k_pooled.Interaction_p  = NaN;
end